<?php
/*--------------------------------------------------------------
   Service.php 2021-06-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview;

use Gambio\Admin\Modules\Statistics\App\Overview\Data\Repository;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\WidgetDefinitions;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Widgets;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\Widget;
use Gambio\Admin\Modules\Statistics\Services\StatisticsOverviewService;

class Service implements StatisticsOverviewService
{
    private const DEFAULT_LANGUAGE_CODE = "de";
    
    private const DEFAULT_WIDGET_WIDTH = 400;
    
    private const DEFAULT_WIDGET_HEIGHT = 200;
    
    private const DEFAULT_WIDGET_HORIZONTAL_POSITION = 0;
    
    private const DEFAULT_WIDGET_VERTICAL_POSITION = 0;
    
    /**
     * @var Repository
     */
    private $repository;
    
    /**
     * @var Factory
     */
    private $factory;
    
    
    /**
     * Constructor.
     */
    public function __construct(Repository $repository, Factory $factory)
    {
        $this->repository = $repository;
        $this->factory    = $factory;
    }
    
    
    /**
     * @inheritDoc
     */
    public function getWidgets(): Widgets
    {
        return $this->repository->getWidgets();
    }
    
    
    /**
     * @inheritDoc
     */
    public function getWidgetById(int $id): Widget
    {
        return $this->repository->getWidgetById($this->factory->createId($id));
    }
    
    
    /**
     * @inheritDoc
     */
    public function getWidgetDefinitions(): WidgetDefinitions
    {
        return $this->repository->getWidgetDefinitions();
    }
    
    
    /**
     * @inheritDoc
     */
    public function addWidget(string $type): void
    {
        $widgetType = $this->factory->createType($type);
        $widgetDefinition = $this->repository->getWidgetDefinitionByType($widgetType);
        
        $this->repository->addWidget(
            $this->factory->createWidgetDatabaseEntry(
               $widgetType,
               $this->factory->createLabel(
                   $widgetDefinition->titles()->getByLanguageCode(self::DEFAULT_LANGUAGE_CODE)->value()
               ),
               $this->factory->createWidth(self::DEFAULT_WIDGET_WIDTH),
               $this->factory->createHeight(self::DEFAULT_WIDGET_HEIGHT),
               $this->factory->createHorizontalPosition(self::DEFAULT_WIDGET_HORIZONTAL_POSITION),
               $this->factory->createVerticalPosition(self::DEFAULT_WIDGET_VERTICAL_POSITION),
               $this->factory->useOptions()->createUpdateSetFromOptions($widgetDefinition->options())
            )
        );
    }
    
    
    /**
     * @inheritDoc
     */
    public function resizeWidget(int $id, int $width, int $height): void
    {
        $this->repository->resizeWidget($this->factory->createId($id),
                                        $this->factory->createWidth($width),
                                        $this->factory->createHeight($height));
    }
    
    
    /**
     * @inheritDoc
     */
    public function moveWidget(int $id, int $horizontalPosition, int $verticalPosition): void
    {
        $this->repository->moveWidget($this->factory->createId($id),
                                      $this->factory->createHorizontalPosition($horizontalPosition),
                                      $this->factory->createVerticalPosition($verticalPosition));
    }
    
    
    /**
     * @inheritDoc
     */
    public function configureWidget(int $id, array $updateSet): void
    {
        $this->repository->configureWidget($this->factory->createId($id),
                                           $this->factory->useOptions()->createUpdateSet($updateSet));
    }
    
    
    /**
     * @inheritDoc
     */
    public function deleteWidget(int $id): void
    {
        $this->repository->deleteWidget($this->factory->createId($id));
    }
    
    
    /**
     * @inheritDoc
     */
    public function renameWidget(int $id, string $label): void
    {
        $this->repository->renameWidget($this->factory->createId($id), $this->factory->createLabel($label));
    }
}